/*
HP21 Calculator Emulator - hpfunc.js
Copyright (c) 2012-2015 Greg Sydney-Smith

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.

Revision History:
2015-03-29 Release 1.07, Updated license terms.
2013-01-01 Updated - mainly to use calcState class.
2012-03-13 Rm displayX() call.
2012-03-13 Initial release.
*/

function abs(n) { return (n<0) ? -n : n; }

// Used to get X before doing a calc on it. Saves it in the LastX register.
function saveLastX() {
  cs.lastX= ib.get();
  return cs.lastX;
}

function HMS2H(n) {
  var a, h, m, s;

  a= (n<0) ? -n : n;
  a *= 10000; // a = Math.floor(a);
  s = a % 100; a /= 100; a= Math.floor(a);
  m = a % 100; a /= 100; a= Math.floor(a);
  h = a + (m/60) + (s/3600);
  return (n<0) ? -h : h;
}

function H2HMS(n) {
  var a, h, m, s;

  a= (n<0) ? -n : n;
  a *= 3600;
  s= a%60; a/=60; a= Math.floor(a);
  m= a%60; a/=60; a= Math.floor(a);
  h= a + (m/100) + (s/10000);
  return (n<0) ? -h : h;
}

function getCalcFactor() {
  return (cs.angleMode == "DEG" ) ? (Math.PI/180) :
         (cs.angleMode == "GRAD") ? (Math.PI/200) : 1;
}

// Input Functions (0-9,.,CHS,EEX) //

function do_CHS() {
  ib.press("-");
}

function do_num(n) {
  if (cs.auto_enter) { do_enter(); }
  ib.press(n);
  cs.flag[3]= 1; // dont use do_sf(3,1) as it calls end_input
}
  
function do_decimal() {
  if (cs.auto_enter) { do_enter(); }
  ib.press(".");
}

function do_EEX() {
  if (cs.auto_enter) { do_enter(); }
  ib.press("e");
  cs.flag[3]= 1; // dont use do_sf(3,1) as it calls end_input
}

function doFunc(code)
{
  switch (code) {
  }
}
 
function do_R2P() {
  x= ib.get();
  var r  = Math.sqrt(x*x + y*y);
  var ang= Math.atan2(y,x) / getCalcFactor();
  y= ang;
  ib.set(r);
}

function do_P2R() {
  var r= ib.get();
  var ang= y;
  x = r*Math.cos(ang * getCalcFactor());
  y = r*Math.sin(ang * getCalcFactor());
  ib.set(x);
}

function do_R2D() {
  x= saveLastX();
  x= x*180/Math.PI;
  ib.set(x);
}

function do_D2R() {
  x= saveLastX();
  x= x/180*Math.PI;
  ib.set(x);
}

function do_HMS2H() {
  x= saveLastX();
  x= HMS2H(x);
  ib.set(x);
}

function do_H2HMS() {
  x= saveLastX();
  x= H2HMS(x);
  ib.set(x);
}

function do_HMSPlus() {
  x= saveLastX();
  var a= HMS2H(x) + HMS2H(y);
  x= H2HMS(a);
  y= z;
  z= t;
  ib.set(x);
}

function do_DSP(n) {
  x= ib.get();
  if (n == "i") n=cs.mem[IREG];
  n= abs(parseInt(n,10))
  if (n > 9) { cs.error=1; return; }
  cs.dsp= n;
  ib.set(x); // Also sets auto_enter (else could use ib.displayX())
}

// round x to the displayed precision
function do_rnd() {
  x= saveLastX();
  var precision= Math.pow(10,cs.dsp);
  var v= (x<0) ? -x : x;
  v *= precision; v += 0.5; v=Math.floor(v); v /= precision;
  x= (x<0) ? -v : v;
  ib.set(x);
}


function do_clx() {
  x= ib.get();
  ib.set(0);
  cs.auto_enter = 0; // next input will overwrite x
}

function do_clr() { // hp21
  y = 0;
  z = 0;
  t = 0;
  cs.mem[0] = 0;
  do_fmt("FIX");
  do_DSP(2);
  do_clx();
}

function do_enter() {
//alert("before: x="+x+" y="+y+" z="+z+" t="+t);
  x= ib.get(); t=z; z=y; y=x;
  ib.set(x);
  cs.auto_enter = 0;
//alert("after: x="+x+" y="+y+" z="+z+" t="+t);
}

function do_rdn() {
  x= ib.get(); var temp=x; x=y; y=z; z=t; t=temp;
  ib.set(x);
}

function do_rup() {
  x= ib.get(); var temp=t; t=z; z=y; y=x; x=temp;
  ib.set(x);
}

function do_xy() {
  x= ib.get(); var temp=x; x=y; y=temp;
  ib.set(x);
}

function do_xi() {
  x= ib.get(); var temp=x; x=cs.mem[IREG]; cs.mem[IREG]=temp;
  ib.set(x);
}

function do_sto(op,mem) {
  x=ib.get();
  if ((mem<0) || (cs.mem.length<=mem)) { cs.error=1; return; }
  switch (op) {
  case "-" : cs.mem[mem] -= x; break;
  case "+" : cs.mem[mem] += x; break;
  case "*" : cs.mem[mem] *= x; break;
  case "/" : cs.mem[mem] /= x; break;
  default  : cs.mem[mem]  = x; break;
  }
  ib.set(x);
}

function do_rcl(mem) {
  if (cs.auto_enter || ib.hasData()) do_enter(); // lift stack unless after ENTER or CLx
  if ((mem<0) || (cs.mem.length<=mem)) { cs.error=1; return; }
  x = cs.mem[mem];
  ib.set(x);
}

function do_pi() {
  if (cs.auto_enter || ib.hasData()) do_enter(); // lift stack unless after ENTER or CLx
  x = 3.141592654;
  ib.set(x);
}

function do_add() {
  x = saveLastX();
  x = y + x;
  y = z;
  z = t;
  ib.set(x);
}

function do_subtract() {
  x = saveLastX();
  x = y - x;
  y = z;
  z = t;
  ib.set(x);
}

function do_multiply() {
  x = saveLastX();
  x = y * x;
  y = z;
  z = t;
  ib.set(x);
}

function do_divide() {
  x = saveLastX();
  x = y / x;
  y = z;
  z = t;
  ib.set(x);
}

function do_sqrt() {
  x = saveLastX();
  x = Math.sqrt(x);
  ib.set(x);
}

function do_pow() {
  x = saveLastX();
  x = Math.pow(x, y);
  y = z;
  z = t;
  ib.set(x);
}

function do_y2x() {
  x = saveLastX();
  x = Math.pow(y, x);
  y = z;
  z = t;
  ib.set(x);
}

function do_epow() {
  x = saveLastX();
  x = Math.pow(Math.E, x);
  ib.set(x);
}

function do_10pow() {
  x = saveLastX();
  x = Math.pow(10, x);
  ib.set(x);
}

function do_pow2() {
  x = saveLastX();
  x = Math.pow(x, 2);
  ib.set(x);
}

function do_ln() {
  x = saveLastX();
  x = Math.log(x);
  ib.set(x);
}

function do_log() {
  x = saveLastX();
  x = Math.LOG10E * Math.log(x);
  ib.set(x);
}

function do_inv() {
  x = saveLastX();
  x = 1/x;
  ib.set(x);
}

// DEG, RAD, GRAD
function do_angle(ang) {
  x= ib.get();
  cs.angleMode = ang;
  ib.set(x);
}

// FIX, SCI, ENG
function do_fmt(fmt) {
  x= ib.get();
  cs.fmt= fmt;
  ib.set(x);
}

function do_sin() {
  x = saveLastX();
  x = Math.sin(x * getCalcFactor());
  x = Math.round(x*10000000000)/10000000000;
  ib.set(x);
}

function do_cos() {
  x = saveLastX();
  x = Math.cos(x * getCalcFactor());
  x = Math.round(x*10000000000)/10000000000;
  ib.set(x);
}

function do_tan() {
  x = saveLastX();
  x = Math.tan(x * getCalcFactor());
  x = Math.round(x*10000000000)/10000000000;
  ib.set(x);
}

function do_asin() {
  x = saveLastX();
  x = Math.asin(x) / getCalcFactor();
  x = Math.round(x*10000000000)/10000000000;
  ib.set(x);
}

function do_acos() {
  x = saveLastX();
  x = Math.acos(x) / getCalcFactor();
  x = Math.round(x*10000000000)/10000000000;
  ib.set(x);
}

function do_atan() {
  x = saveLastX();
  x = Math.atan(x) / getCalcFactor();
  x = Math.round(x*10000000000)/10000000000;
  ib.set(x);
}

