/*
HP21 Calculator Emulator - hpinput.js
Copyright (c) 2012-2015 Greg Sydney-Smith

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.

Revision History:
2015-03-29 Release 1.07, Updated license terms.
2013-01-01 From HP29 version.
2012-12-24 Initial release.
*/

/*
 A Register consists of 14 BCD digits plus, in this implementation, a dot.
 The calculator internals were all 10+2 - 10 digit mantissa and 2 digit exponent
 This required 14 digits: mantissa sign (0 or 9), mantissa, exponent sign, exponent
 They used another "CPU" register to hold a mask which blanked out display digits
 (eg the spaces after "0.00" in DSP 2 or after "12." during data entry) and which
 recorded where the decimal point should be.
 In this implementation I just use space and dot in the register.
 The display was less than 15 digits for the HP21/25/29 but the register still
 held 14 BCD digits. This code shouldn't change between the calculators.  
*/

// hp21-29 version

function InputBuffer()
{
  var that = this;
  var hasDot = false;
  var nxt= 1;
  var state = "NORMAL";
  this.d = [];

  s2a(" 0.000000000 00");

  function a2s(pos,len) {
    var s="";
    for (var i=0; i<len; i++)
      s += that.d[pos+i];
    return s;
  }
  
  function s2a(s) {
    for (var i=0; i<1+10+1+1+2; i++)
        that.d[i]= s.substr(i,1);
  }

  this.get = function () {
    state= "NORMAL"; nxt=1;
    // Convert the display digits for 'd' into the JavaScript number 'x'.
    // e.g.  ' 06.0' => 6
    var x   = parseFloat(a2s( 0,12));  if (isNaN(x)) x= 0;
    var exp = parseInt(  a2s(12, 3),10);
    if (!isNaN(exp)) x *= Math.pow(10, exp);
    return x;
  };
  
  this.set = function (v) {

    // Basics
    if (isNaN(v) || v == Number.NEGATIVE_INFINITY || v == Number.POSITIVE_INFINITY) {
      cs.error=1;
      s2a(" 0.000000000 00");
      return;
    }
    // hp overflow handling
    var MaxFloat = 9.999999999 * Math.pow(10, 99);
    if (v >=  MaxFloat) v =  MaxFloat; // positive overflow is written as maximum number
    if (v <= -MaxFloat) v = -MaxFloat; // negative overflow is written as minimum number
    
    var v1  = (v<0) ? -v : v;
    var exp = (v1 == 0) ? 0 : Math.floor(Math.log(v1) / Math.LN10);
    v1 /= Math.pow(10,exp); // 0 or 1.00 ... 9.99
    var s = ""+v1;
    if (s.indexOf(".") < 0) s += ".";
    s += "0000000000";

    // set sign digit
    this.d[0]= (v<0) ? "-" : " ";

    // set mantissa
    for (i=0; i<11; i++)
      this.d[1+i]= s.substr(i,1);

    // set exponent
    var e =  (exp<0) ? -exp : exp;
    this.d[12]= (exp<0) ? "-" : " ";
    this.d[13]= ""+Math.floor(e/10);
    this.d[14]= ""+(e%10);

    x=v;
    cs.auto_enter=1;
  };

  this.press = function (n) {
    var type = ("0" <= n && n<= "9") ? "n" : (n==".") ? "." : (n=="e") ? "e" : "-";
    switch (state+type) {
    case "NORMALn" : s2a("               "); nxt=1; hasDot= false; this.d[nxt++]= n;   this.d[nxt]= "."; state="INP"; break;
    case "NORMAL." : s2a("               "); nxt=1; hasDot= true;  this.d[nxt++]= "."; state="INP"; break;
    case "NORMALe" : s2a(" 1.          00"); state="EEX"; break;
    case "NORMAL-" : this.d[0] = (this.d[0]==" ") ? "-" : " "; break;
    case "INPn"    : if (nxt >= (hasDot?12:11)) break;
                     this.d[nxt++]= n;
                     if (!hasDot) this.d[nxt  ]= '.';
                     break;
    case "INP."    : if (!hasDot) this.d[nxt++]= '.';
                     hasDot=true;
                     break;
    case "INPe"    : if (nxt >= 12+1-3) break; // <<< physical display size+dp-3
                     state="EEX"; this.d[13]= "0"; this.d[14]= "0"; break;
    case "INP-"    : this.d[0] = (this.d[0]==" ") ? "-" : " "; break;
    case "EEXn"    : this.d[13]= this.d[14]; this.d[14]= n; break;
    case "EEX."    : break;
    case "EEXe"    : break;
    case "EEX-"    : this.d[12]= (this.d[12]=='-') ? " " : "-"; break;
    }
  }
  
  this.hasData    = function () { return (state != "NORMAL"); };
  this.display    = function () { display(this.d,state=="EEX"); };
}


