"use strict";

import {resize, Slider, Color, addLabel, DarkMode, TextBox, Select, resizeAndCenterX} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import DirectionalLight from "../../materials/DirectionalLight.js";
import {Sphere, Cube} from "../../geometry/Figure.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";
import Teapot from "../../geometry/Teapot.js";
import Bunny from "../../geometry/Bunny.js";

/**
 * Fuente de luz direccional.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    
    let sphereFigure = new Sphere(1, 30,30);
    let cubeFigure = new Cube(1.5);
    let teapotFigure = new Teapot();
    let bunnyFigure = new Bunny();
    
    let lightDirection = [0, -2, 0, 0];
    let material = { ka: [0.329412, 0.223529, 0.027451], kd: [0.780392, 0.568627, 0.113725], ks: [0.992157, 0.941176, 0.807843], 
                     shininess: 0.21794872*128};
    let light = { direction: lightDirection, la: [.3, .3, .3], ld: [1,1,1], ls: [1.0, 1.0, 1.0]};
    let uniforms = { u_light: light, u_material: material};  
    let translate = M4.translate(0,.5,0);
    let figures = { 
        tetera: new GeometryPainter(gl, new DirectionalLight(gl, WebGL, teapotFigure, uniforms), M4.multiply(M4.scale(.9,.9,.9), translate)),
        cubo: new GeometryPainter(gl, new DirectionalLight(gl, WebGL, cubeFigure, uniforms), translate) ,
        esfera : new GeometryPainter(gl, new DirectionalLight(gl, WebGL, sphereFigure, uniforms), translate),
        conejo: new GeometryPainter(gl, new DirectionalLight(gl, WebGL, bunnyFigure, uniforms), translate)};

    let selectedFigure = figures.tetera;  
   
    var fov = 102;
    let zNear = .1;
    let zFar = 100;
    var pos = new V3(0,1.2,2.7);
    let camera = new TrackballCamera(pos);
    camera.setZoomConfig(4,1.5);
    
    let light_controls = document.getElementById("ui-container-center");
    addLabel(light_controls, "Dirección de la luz");
    var light_dir_x = new Slider(light_controls, "x", -10, 10, upadeteLDir(0), lightDirection[0], 0.01, ["#4994D0"]);
    var light_dir_y = new Slider(light_controls, "y", -10, 10, upadeteLDir(1), lightDirection[1], 0.01, ["#4994D0"]);
    var light_dir_z = new Slider(light_controls, "z", -10, 10, upadeteLDir(2), lightDirection[2], 0.01, ["#4994D0"]);
    
    let darkMode = new DarkMode(draw, [light_controls]);

    let ccLabel = new TextBox(container, { left: 15, top: 4, type: "px"},"", "author-small");

    /* Selects */
    let right_controls = document.getElementById("right-options");
    let selectFigure = new Select(right_controls, "figures", setFigure, figures);
   
    let modelViewMatrix;

    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.CULL_FACE);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    draw();

    function draw(){        
        resize(canvas);
        resizeAndCenterX(light_controls, {bottom:8});
        var {width, height} = gl.canvas;
        
        darkMode.check(gl);

        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

        let aspect = width/height;
        var projectionMatrix = M4.perspective(fov, aspect, zNear, zFar);

        let viewMatrix = camera.getMatrix();
        var lightDir = M4.multiplyVector(viewMatrix, lightDirection);
        
        uniforms = { u_light: light, u_material: material}; 
        uniforms.u_light.direction = lightDir;

        for(var i = 0; i < 9; i++){
            if(i < 3)
                modelViewMatrix = M4.multiply(viewMatrix, M4.translate(-i*3 + 3, -1, -6));
            else if (i < 6)
                modelViewMatrix = M4.multiply(viewMatrix, M4.translate((3-i)*3 + 3, -1, -3));
            else
                modelViewMatrix = M4.multiply(viewMatrix, M4.translate((6-i)*3 + 3, -1, 0));

            selectedFigure.draw(gl, modelViewMatrix, projectionMatrix, uniforms);       
        }
    }
    function setFigure(value){
        if(value == 'conejo'){
            ccLabel.setText( "<i>Bunny</i>. Modelo obtenido de <i>Stanford <br> University Computer Graphics Laboratory</i>.");
        }else
            ccLabel.setText("");
        selectedFigure = figures[value];
        draw();
    }
    function upadeteLDir(index){
        return function(value){
            lightDirection[index] = value;
            draw();
        }
    }
}