"use strict";

import {resize, Slider, Color, addLabel, resizeAndCenterX, DarkMode, TextBox, Select} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import Phong from "../../materials/Phong.js";
import {Sphere, Cube} from "../../geometry/Figure.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";
import Teapot from "../../geometry/Teapot.js";
import Bunny from "../../geometry/Bunny.js";
import Light from "../../geometry/Light.js";

/**
 * Modelo de Iluminación de Phong.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    
    let sphereFigure = new Sphere(1, 30,30);
    let cubeFigure = new Cube(1.5);
    let teapotFigure = new Teapot();
    let bunnyFigure = new Bunny();
    
    let lightPosition = [1.2, 2.5, -1.3, 0];
    let material = { ka: 0.3, kd: 1, ks: 1, shininess: 20};
    let light = { position: lightPosition, la: [0.690, 0.690, 0.690], ld: [1.0, 1.0, 1.0],
                  ls: [1.0, 1.0, 1.0]};
    let uniforms = { u_light: light, u_material: material};  
    let translate = M4.translate(0,.5,0);
    let figures = { 
        tetera: new GeometryPainter(gl, new Phong(gl, WebGL, teapotFigure, uniforms), translate),
        esfera : new GeometryPainter(gl, new Phong(gl, WebGL, sphereFigure, uniforms), translate),
        conejo: new GeometryPainter(gl, new Phong(gl, WebGL, bunnyFigure, uniforms), translate),
        cubo: new GeometryPainter(gl, new Phong(gl, WebGL, cubeFigure, uniforms), translate)};

    let selectedFigure = figures.tetera;  
   
    var fov = 110;
    let zNear = .1;
    let zFar = 100;
    var pos = new V3(0,1.4,2.7);
    let camera = new TrackballCamera(pos);
    
    let controls = document.getElementById("ui-container");
    controls.className = "big-bottom-box";
    let c_first = document.getElementById("first");
    let c_second = document.getElementById("second");
    let c_third = document.getElementById("third");

    /* Luz Ambiental */
    var kA_slider = new Slider(c_first, "k<sub>A</sub>", 0, 1, upadeteK(0), material.ka, 0.01, ["#4994D0"], "", true);
    var ambient_color = new Color(c_first, "L<sub>A</sub>", setColor(0), "#b0b0b0");
    /* Luz Difusa */
    var kd_slider = new Slider(c_second, "k<sub>D</sub>", 0, 1, upadeteK(1), material.kd, 0.01, ["#4994D0"], "", true);
    var diffuse_color = new Color(c_second, "L<sub>D</sub>", setColor(1), "#FFFFFF");
    /* Luz Especular */
    var ks_slider = new Slider(c_third, "k<sub>S</sub>", 0, 1, upadeteK(2), material.ks, 0.01, ["#4994D0"], "", true);
    var specular_color = new Color(c_third, "L<sub>S</sub>", setColor(2), "#FFFFFF");
    var shininess_slider = new Slider(c_third, "α", .1, 300, upadeteShininess, material.shininess, 0.1, ["#4994D0"], "", true);

    let light_controls = document.getElementById("ui-container-up");
    addLabel(light_controls, "Posición de la luz");
    var light_pos_x = new Slider(light_controls, "x", -10, 10, upadeteLPos(0), lightPosition[0], 0.01, ["#4994D0"]);
    var light_pos_y = new Slider(light_controls, "y", -10, 10, upadeteLPos(1), lightPosition[1], 0.01, ["#4994D0"]);
    var light_pos_z = new Slider(light_controls, "z", -10, 10, upadeteLPos(2), lightPosition[2], 0.01, ["#4994D0"]);
    
    let all_controls = [controls, light_controls];
    let darkMode = new DarkMode(draw, all_controls);

    let ccLabel = new TextBox(container, { left: 20, top: 55, type: "px"}, "", "author-small");

    /* Selects */
    let right_controls = document.getElementById("right-options");
    let selectFigure = new Select(right_controls, "figures", setFigure, figures);
    //Luz
    let bulb = new Light(gl, WebGL, lightPosition);
    
    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.CULL_FACE);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    draw();
    
    function draw(){        
        resize(canvas);
        resizeAndCenterX(light_controls, {});
        var {width, height} = gl.canvas;
        
        darkMode.check(gl);

        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

        let aspect = width/height;
        var projectionMatrix = M4.perspective(fov, aspect, zNear, zFar);

        let viewMatrix = camera.getMatrix();
        var lightPos = M4.multiplyVector(viewMatrix, lightPosition);
       
        bulb.draw(gl, viewMatrix, projectionMatrix, lightPosition, [light.ld[0], light.ld[1], light.ld[2], 1]);
        
        uniforms = { u_light: light, u_material: material}; 
        uniforms.u_light.position = lightPos;
 
        selectedFigure.draw(gl, viewMatrix, projectionMatrix, uniforms);       
    }
    function setFigure(value){
        if(value == 'conejo'){
            ccLabel.setText( "<i>Bunny</i>. Modelo obtenido de <i>Stanford <br> University Computer Graphics Laboratory</i>.");
        }else
            ccLabel.setText("");
        selectedFigure = figures[value];
        draw();
    }
    function upadeteShininess(value){
        material.shininess = value;
        draw();
    }
    function upadeteK(index){
        return function(value){
            switch(index){
                case 0:
                    material.ka = value;
                    break;
                case 1:
                    material.kd = value;
                    break;
                case 2:
                    material.ks = value;
                    break;
            }
            draw();
        }
    }
    function setColor(index){
        return function(value){
            switch(index){
                case 0:
                    light.la = value;
                    break;
                case 1:
                    light.ld = value;
                    break;
                case 2:
                    light.ls = value;
                    break;
            }
            draw();
        }
    }
    function upadeteLPos(index){
        return function(value){
            lightPosition[index] = value;
            draw();
        }
    }
}