import {getDisconnectedVertices, getNormalsFromIndices, getNormalsFromVertices} from "../geometry/Figure.js";

/**
 * Programa de sombreado BadNormalMapping.
 * @author Melissa Méndez Servín.
 */
export default class BadNormalMapping{
    
    constructor(gl, WebGL, figure, srcTextures, draw_callback, initialUniforms){
        var vsh = `#version 300 es
                    uniform mat4 u_PVM_matrix;
                    uniform mat4 u_VM_matrix;
                    uniform mat4 u_VMN_matrix;
                    
                    in vec4 a_position;
                    in vec3 a_normal;
                    in vec2 a_texcoord;
                
                    out vec3 v_normal; 
                    out vec3 v_position; 
                    out vec2 v_texcoord;

                    void main(){
                    
                        v_position = (u_VM_matrix * a_position).xyz;
                        v_normal = (u_VMN_matrix * vec4(a_normal, 0)).xyz;
                        v_texcoord = a_texcoord;
            
                        gl_Position = u_PVM_matrix * a_position;
                    }`;
        var fsh = `#version 300 es
                    precision highp float;
                    
                    struct Material{    
                        sampler2D diffuse_map;
                        sampler2D normal_map;
                        float shininess;
                    };
                    
                    struct Light{
                        vec4 position;
                        vec3 la;
                        vec3 ld;
                        vec3 ls;
                    };

                    uniform Light u_light;
                    uniform Material u_material; 

                    in vec3 v_normal;
                    in vec3 v_position;
                    in vec2 v_texcoord;

                    out vec4 glColor;
                
                    void main(){
                        vec3 L = normalize(u_light.position.xyz - v_position);
                        vec3 N = texture(u_material.normal_map, v_texcoord).rgb * 2.0 + 1.0;
                        N = normalize(N);

                        float cos_angle = max(dot(N, L), 0.0);
                        vec3 diffuse_color = texture(u_material.diffuse_map, v_texcoord).rgb;
                        
                        float specular = 0.0;
                        
                        if(cos_angle > 0.0){
                            vec3 R = reflect(-L,N);
                            vec3 V = normalize(-v_position);
                            float spec_angle = max(dot(R,V),0.0);
                            specular = pow(spec_angle, u_material.shininess);
                        }
                        float d = length(u_light.position.xyz - v_position);
                        float attenuation = 1.0/(0.01*d*d + 0.22*d + 0.03);

                        vec3 contribution = vec3(diffuse_color * u_light.la + 
                                            attenuation * (diffuse_color * u_light.ld * cos_angle +
                                                           u_light.ls * specular));
                        glColor = vec4(contribution, 1.0);
                    }`; 

        if (WebGL.programs["BadNM"])
            this.program = WebGL.programs["BadNM"];
        else
            this.program = WebGL.createProgram(gl, vsh, fsh, "BadNM");
            
        this.vertices = (figure.byIndices || figure.getNormals) ?  figure.getVertices() : getDisconnectedVertices(figure.getVertices(), figure.getFaces());
        if(figure.getNormals)
            this.normals = figure.getNormals();
        else
            this.normals = (figure.byIndices) ? getNormalsFromIndices(figure.getFaces(), this.vertices) : getNormalsFromVertices(this.vertices);
        
        let attributes = {  position: { numComponents: 3, 
                                                 data: this.vertices},
                              normal: { numComponents: 3, 
                                                 data: this.normals},
                            texcoord: { numComponents: 2, 
                                                 data: figure.getTexCoordinates(),
                                            normalize: true}
                         };
        
        if(figure.byIndices){
            let vaoAndIndices = WebGL.setVAOAndAttributes(gl, this.program, attributes, figure.getFaces());
            this.vao = vaoAndIndices.vao;
            this.indexBuffer = vaoAndIndices.indexBuffer;
            this.numIndices = figure.numIndices;
        }else{
            this.vao = WebGL.setVAOAndAttributes(gl, this.program, attributes);  
            this.numElements = this.vertices.length/3;
        }
        this.setUniforms = WebGL.setUniforms(gl, this.program);

        this.texture = WebGL.loadImages( srcTextures, createTextures);
        this.uniforms = Object.assign({}, initialUniforms);

        function createTextures(images){
            WebGL.createTextures(gl, images, srcTextures, draw_callback);
        }
    }
}