import {getDisconnectedVertices, getNormalsFromIndices, getNormalsFromVertices} from "../geometry/Figure.js";

/**
 * Programa de sombreado para simular una fuente de luz puntual, 
 * usando el modelo de ilumación de Phong con materiales.
 * @author Melissa Méndez Servín.
 */
export default class PointLight{

    constructor(gl, WebGL, figure, initialUniforms){
        var vsh = `#version 300 es
                    uniform mat4 u_PVM_matrix;
                    uniform mat4 u_VM_matrix;
                    uniform mat4 u_VMN_matrix;
                    
                    in vec4 a_position;
                    in vec3 a_normal;
                
                    out vec3 v_normal; 
                    out vec3 v_position; 

                    void main(){
                    
                        v_position = (u_VM_matrix * a_position).xyz;
                        v_normal = (u_VMN_matrix * vec4(a_normal, 0)).xyz;
            
                        gl_Position = u_PVM_matrix * a_position;
                    }`;
        var fsh = `#version 300 es
                    precision highp float;
                    
                    struct Material{
                        vec3 ka;
                        vec3 kd;
                        vec3 ks;
                        float shininess;
                    };
                    struct Light{
                        vec4 position;
                        vec3 la;
                        vec3 ld;
                        vec3 ls;
                        
                        float quadratic;
                        float linear;
                        float constant;
                    };

                    uniform Light u_light;
                    uniform Material u_material; 

                    in vec3 v_normal;
                    in vec3 v_position;
                    
                    out vec4 glColor;
                
                    void main(){
                        vec3 L = normalize(u_light.position.xyz - v_position);
                        vec3 N = normalize(v_normal);
            
                        float cos_angle = max(dot(N, L), 0.0);
                        float specular = 0.0;

                        if(cos_angle > 0.0){
                            vec3 R = reflect(-L,N);
                            vec3 V = normalize(-v_position);
                            float spec_angle = max(dot(R,V),0.0);
                            specular = pow(spec_angle, u_material.shininess);
                        }
                        float d = length(u_light.position.xyz - v_position);
                        float attenuation = 1.0/(u_light.quadratic*d*d + u_light.linear * d + u_light.constant);

                        vec3 contribution = vec3(u_material.ka * u_light.la + 
                                                 attenuation * (u_material.kd * u_light.ld * cos_angle +
                                                                u_material.ks * u_light.ls * specular));

                        glColor = vec4(contribution, 1.0);
                    }`; 

        if (WebGL.programs["PointL"])
            this.program = WebGL.programs["PointL"];
        else
            this.program = WebGL.createProgram(gl, vsh, fsh, "PointL");
            
        this.vertices = (figure.byIndices || figure.getNormals) ?  figure.getVertices() : getDisconnectedVertices(figure.getVertices(), figure.getFaces());
        if(figure.getNormals)
            this.normals = figure.getNormals();
        else
            this.normals = (figure.byIndices) ? getNormalsFromIndices(figure.getFaces(), this.vertices) : getNormalsFromVertices(this.vertices);
        
        let attributes = {  position: { numComponents: 3, 
                                                 data: this.vertices},
                              normal: { numComponents: 3, 
                                                 data: this.normals}
                         };
        
        if(figure.byIndices){
            let vaoAndIndices = WebGL.setVAOAndAttributes(gl, this.program, attributes, figure.getFaces());
            this.vao = vaoAndIndices.vao;
            this.indexBuffer = vaoAndIndices.indexBuffer;
            this.numIndices = figure.numIndices;
        }else{
            this.vao = WebGL.setVAOAndAttributes(gl, this.program, attributes);  
            this.numElements = this.vertices.length/3;
        }
        this.setUniforms = WebGL.setUniforms(gl, this.program);
        this.uniforms = Object.assign({}, initialUniforms);
    }
}