"use strict";

import {resize, Slider} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import V3 from "../../modules/V3.js";
import GridXZ from "../../geometry/GridXZ.js";
import DirectionalLight from "../../materials/DirectionalLight.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import { Cone, Cube, Sphere} from "../../geometry/Figure.js";
import Vector3D from "../../geometry/Vector3D.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";

/**
 * Transformación de la cámara (Animación).
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    
    //Figuras
    let gridXZ = new GridXZ(gl, 68, 0, null);

    let figuresGeom = [new Cube(1), new Sphere(.5, 25, 25), new Cone(.5, 1, 4, false)];
     
    let lightDirection = [0, -4, -2, 0];
    let lightPosition = [0, 2, 0, 0];
    let colors = [[0.780392, 0,0], [0.25, 0.95,0.8], [1, 1,0]];
    let material = { ka: [0.2,.2,.2], kd: [0.780392, 0,0], ks: [0.992157, 0.941176, 0.807843], 
                    shininess: 0.21794872*128};
    let light = { direction: lightDirection, la: [.5, .5, .5], ld: [1,1,1], ls: [1.0, 1.0, 1.0]};
    let uniforms = { u_light: light, u_material: material};  

    //Objetos a renderizar 
    let objects = [];
    for(var i= 0; i< figuresGeom.length; i++)
        objects.push(new GeometryPainter(gl, new DirectionalLight(gl, WebGL, figuresGeom[i], uniforms)));                                   
    
    let axis = [ new Vector3D(gl, WebGL, new V3(1,0,0), null, .15, [0,.2,1,1], lightPosition),
                 new Vector3D(gl, WebGL, new V3(0,1,0), null, .15, [0,1,0.2,1], lightPosition),
                 new Vector3D(gl, WebGL, new V3(0,0,1), null, .15, [1,.2,0,1], lightPosition)];
    
     
    resize(canvas);
    
    let zNear = .1;
    let zFar = 200;

    let camera = new TrackballCamera(new V3(0,1.2,2));
    
    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.SCISSOR_TEST);
    gl.clearColor(0, 0, 0, 0);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    draw();


    function draw(){
        resize(canvas);
        
        var {width, height} = gl.canvas;
        const smallWidth = width / 3;
        
        let aspect = smallWidth/height;
        var projectionMatrix = M4.perspective(90, aspect, zNear, zFar);
       
        var viewMatrix = camera.getMatrix();
        var lightDir = M4.multiplyVector(viewMatrix, lightDirection);
        var lightPos = M4.multiplyVector(viewMatrix, lightPosition);
        
        //Izquierda
        renderLocalScene(0, smallWidth, height, viewMatrix, projectionMatrix, lightDir, lightPos);
        //Medio
        renderLocalScene(1, smallWidth, height, viewMatrix, projectionMatrix, lightDir, lightPos);
        //Derecha
        renderLocalScene(2, smallWidth, height, viewMatrix, projectionMatrix, lightDir, lightPos);
       
    }
    function renderLocalScene(id, width, height, viewMatrix, projectionMatrix, lightDir, lightPos){
        var offset = id * width * 0.05;
        var smallWidth = width * .9;
        gl.viewport(id * width + offset, 0, smallWidth, height);
        gl.scissor(id * width + offset, 0, smallWidth, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

        gridXZ.draw(gl, viewMatrix, projectionMatrix);

        for(var i=0; i < 3; i++)
            axis[i].draw(gl, viewMatrix, projectionMatrix, lightPos);
        

        uniforms.u_light.direction = lightDir;
        uniforms.u_material.kd = colors[id];
        objects[id].draw(gl, viewMatrix, projectionMatrix, uniforms);
    }

}