"use strict";

import {resize, Slider, Color, addLabel, scaleToFitContainer} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import Phong from "../../materials/Phong.js";
import {Cone, Sphere, UnitSphere} from "../../geometry/Figure.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";
import Light from "../../geometry/Light.js";

/**
 * Modelo de reflexión especular.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    
    let sphereFigure = new Sphere(1, 30,30);
    let lightPosition = [4, 6, 2, 0];
    let material = { ka: 0.2, kd: 0, ks: 1, shininess: 44};
    let light = { position: lightPosition, la: [0.05375, 0.05, 0.06625], ld: [0.18275, 0.17, 0.22525],
                  ls: [1.0, 1.0, 1.0]};
    let diffuseUniforms = { u_light: light, u_material: material}; 
    let figure = new GeometryPainter(gl, new Phong(gl, WebGL, sphereFigure, diffuseUniforms), M4.translate(0,.3,0));
    var fov = 90;
    if(window.innerWidth < window.innerHeight)
        fov = 120;

    let zNear = 1;
    let zFar = 100;
    var pos = new V3(0,1.4,2.3);
    let camera = new TrackballCamera(pos);
    
    let controls = document.getElementById("ui-container");
    controls.className = "big-bottom-box";
    var ks_slider = new Slider(controls, "k<sub>S</sub>", 0, 1, upadeteKS, material.ks, 0.01, ["#4994D0"], "", true);
    var specular_color = new Color(controls, "L<sub>S</sub>", setColor, "#FFFFFF");
    var shininess_slider = new Slider(controls, "α", 1, 300, upadeteShininess, material.shininess, 0.1, ["#4994D0"], "", true);
    let light_controls = document.getElementById("ui-container-up");
    addLabel(light_controls, "Luz");
    var light_pos_x = new Slider(light_controls, "x", -10, 10, upadeteLPos(0), lightPosition[0], 0.01, ["#4994D0"]);
    var light_pos_y = new Slider(light_controls, "y", -10, 10, upadeteLPos(1), lightPosition[1], 0.01, ["#4994D0"]);
    var light_pos_z = new Slider(light_controls, "z", -10, 10, upadeteLPos(2), lightPosition[2], 0.01, ["#4994D0"]);
  
    //Luz
    let bulb = new Light(gl, WebGL, lightPosition);
    
    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.CULL_FACE);
    
    gl.clearColor(0, 0, 0, 0);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    draw();
    

    function draw(){        
        resize(canvas);
        scaleToFitContainer(light_controls, {x:50,y:0});
        var {width, height} = gl.canvas;

        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
             
        let aspect = width/height;
        var projectionMatrix = M4.perspective(fov, aspect, zNear, zFar);

        let viewMatrix = camera.getMatrix();
        var lightPos = M4.multiplyVector(viewMatrix, lightPosition);
        
        bulb.draw(gl, viewMatrix, projectionMatrix, lightPosition);
        
        diffuseUniforms = { u_light: light, u_material: material}; 
        diffuseUniforms.u_light.position = lightPos;
 
        figure.draw(gl, viewMatrix, projectionMatrix, diffuseUniforms);       
    }

    function upadeteShininess(value){
        material.shininess = value;
        draw();
    }
    function upadeteKS(value){
        material.ks = value;
        draw();
    }
    function setColor(value){
        light.ls = value;
        draw();
    }
    function upadeteLPos(index){
        return function(value){
            lightPosition[index] = value;
            draw();
        }
    }
}