import M3 from "./M3.js";

/**
 * Clase auxiliar para el manejo de eventos, en particular con 
 * respecto a la posición y clic del ratón.
 * @author Melissa Méndez Servín.
 */

/**
 * Obtiene las coordenadas en pantalla aproximadas a la posición del punto 
 * dado en el canvas. Considerando que la ventana y el 
 * canvas tienen las mismas medidas
 * (NDC coordinates)[-1,1] = [0,width] y [-1,1] = [0,height]
 * (NDCCoords + 1) * 0.5 => [0,1]
 * @param {*} point el punto en el canvas 
 * @param {*} projectionMatrix  la matriz de proyección que se usa
 * @param {*} width  el ancho del canvas/ventana
 * @param {*} height la altura del canvas/ventana
 */
function getPixelPosition(point, projectionMatrix, width, height, unit) {
    var NDCCoords = M3.multiplyVector(projectionMatrix, [unit * point.x, unit * point.y, 1]);
    var coords = { x: Math.floor((NDCCoords[0] + 1) * 0.5 * width), 
                   y: height - Math.floor((NDCCoords[1] + 1) * 0.5 * height)};
    return coords;
}
/**
 * Obtiene la posición del mouse en el canvas
 * @param {*} evt 
 * @param {*} canvas 
 */
function getMousePositionInCanvas(evt, canvas) {
    const boundary = canvas.getBoundingClientRect();
    return limitPosition(evt.clientX, evt.clientY, boundary);
}
/**
 * Obtiene la posición del toque en la pantalla dentro 
 * del canvas
 * @param {*} e 
 * @param {*} canvas 
 */
function getTouchPositionInCanvas(e, canvas) {
    const boundary = canvas.getBoundingClientRect();
    var evt = (typeof e.originalEvent === 'undefined') ? e : e.originalEvent;
    var touch = evt.touches[0] || evt.changedTouches[0];
    return limitPosition(touch.clientX, touch.clientY, boundary);
}
/**
 * Devuelve la posición dentro del rango [clientX, clientY].
 * @param {*} clientX 
 * @param {*} clientY 
 * @param {*} boundary 
 */
function limitPosition(clientX, clientY, boundary){
    var x = clientX - boundary.left;
    var y = clientY - boundary.top;
    if( x < 0)
        x = 0;
    else if( x > boundary.width)
        x = boundary.width;
    if (y < 0)
        y = 0;
    else if (y > boundary.height)
        y = boundary.height;
    return { x: x, y: y };
}

export{
    getPixelPosition,
    getMousePositionInCanvas,
    getTouchPositionInCanvas
}