"use strict";

import {resize, resizeAndCenterX, Slider} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import MultipleSpotlights from "../../materials/MultipleSpotlights.js";
import {Plane, RectangularPrism} from "../../geometry/Figure.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";

/**
 * Modelo RGB utilizando 3 fuentes de luz puntuales y un plno blanco.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    
    let light_positions = [[0, .1, 3.8, 0], [-.1, -.05, 3.8, 0], [.1, -.05, 3.8, 0]];
    
    let lightDirection = [0, 0, -1, 0];
  
    let material = { ka: [0.0,0.0,0.0], kd: [1, 1, 1], ks: [0, 0, 0], 
                    shininess: 0};

    let cuff_off_angle = cos(2.4);
    let fall_off_factor = 4;
    let quadratic = 0.02;
    let linear =  0.05;
    let constant = 0.0;
    let dir_light = {direction: lightDirection, la: [.1, .1, .1], ld: [0,0,0], ls: [1.0, 1.0, 1.0]};
    let red_light =   { position: light_positions[0], direction: lightDirection, la: [0, 0, 0], ld: [1,0,0], ls: [1.0, 0.0, 0.0],
                        quadratic : quadratic, linear: linear, constant: constant, cut_off: cuff_off_angle, fall_off: fall_off_factor};
    let green_light = { position: light_positions[1], direction: lightDirection, la: [0, 0, 0], ld: [0,1,0], ls: [0.0, 1.0, 0.0],
                        quadratic : quadratic, linear: linear, constant: constant, cut_off: cuff_off_angle, fall_off: fall_off_factor};
    let blue_light =  { position: light_positions[2], direction: lightDirection, la: [0, 0, 0], ld: [0,0,1], ls: [0.0, 0.0, 1.0],
                        quadratic : quadratic, linear: linear, constant: constant, cut_off: cuff_off_angle, fall_off: fall_off_factor};
    
    let spots = [red_light, green_light, blue_light];
    let uniforms = { u_spot_lights: spots, u_material: material, u_dir_light: dir_light};  
      
    //Piso 
    var rect = new Plane(2,2);
    let plane = new GeometryPainter(gl, new MultipleSpotlights(gl, WebGL, rect, uniforms, spots.length )); 

    var fov = 90;
    let zNear = .1;
    let zFar = 100;
    var pos = new V3(0,0,.4);
    let camera = new TrackballCamera(pos);
    
    let controls = document.getElementById("ui-container-center");
    controls.setAttribute("obscure-mode", true);

    var red_slider = new Slider(controls, "R", 0, 1, setRGBValue(0), 1, 0.01, ["#4994D0"], "");
    var green_slider = new Slider(controls, "G", 0, 1, setRGBValue(1), 1, 0.01, ["#4994D0"], "");
    var blue_slider = new Slider(controls, "B", 0, 1, setRGBValue(2), 1, 0.01, ["#4994D0"], "");
   
    gl.enable(gl.DEPTH_TEST);

    window.addEventListener('resize', draw);
    //registerCameraEvents(camera, canvas, draw);
    
    draw();

    function draw(){        
        resize(canvas);
        resizeAndCenterX(controls, {bottom:5});
        var {width, height} = gl.canvas;
        
        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

        let aspect = width/height;
        var projectionMatrix = M4.perspective(fov, aspect, zNear, zFar);

        let viewMatrix = camera.getMatrix();
        
        var lightDir = M4.multiplyVector(viewMatrix, lightDirection);
        uniforms.u_dir_light.direction = lightDir;

        for(var i = 0; i < spots.length; i++){
            var lightPos = M4.multiplyVector(viewMatrix, light_positions[i]);
            uniforms.u_spot_lights[i].position = lightPos;
            uniforms.u_spot_lights[i].direction = lightDir;
        }
        plane.draw(gl, viewMatrix, projectionMatrix, uniforms);
              
    }
    function setRGBValue(index){
        return function(value){
            uniforms.u_spot_lights[index].ld[index] = value;
            uniforms.u_spot_lights[index].ls[index] = value;
            draw();
        }
    }
    
    function cos(degree){
        return Math.cos(degree * Math.PI/180) + 1 - 1;
    }
}