"use strict";

import {resize, Slider, addLabel, resizeAndCenterX, DarkMode, Select, Checkbox} from "../../modules/HTMLUtils.js";
import WebGL from "../../modules/WebGLUtils.js";
import M4 from "../../modules/M4.js";
import NormalMapping from "../../materials/NormalMapping.js";
import PhongTexture from "../../materials/PhongTexture.js";
import {Plane} from "../../geometry/Figure.js";
import {TrackballCamera, registerCameraEvents} from "../../modules/Trackballcamera.js";
import V3 from "../../modules/V3.js";
import GeometryPainter from "../../geometry/GeometryPainter.js";
import Light from "../../geometry/Light.js";

/**
 * Normal mapping.
 * @author Melissa Méndez Servín.
 */
window.addEventListener("load", main);

function main(){
    var canvas = document.getElementById("gl_canvas");
    var gl = WebGL.init(canvas);
    if(!gl) return;
    let planeFigure = new Plane(1,1, false);
    let srcTextures = [ "../../../images/capitulo_07/textures/Bricks071_1K_Color.jpg",
                        "../../../images/capitulo_07/textures/Bricks071_1K_Normal.jpg",
                        "../../../images/capitulo_07/textures/Tiles093_1K_Color.jpg",
                        "../../../images/capitulo_07/textures/Tiles093_1K_Normal.jpg",
                        "../../../images/capitulo_07/textures/Tiles032_1K_Color.jpg",
                        "../../../images/capitulo_07/textures/Tiles032_1K_Normal.jpg",
                        "../../../images/capitulo_07/textures/Gravel022_1K_Color.png",
                        "../../../images/capitulo_07/textures/Gravel022_1K_Normal.png"
                    ];
    
    let lightPosition = [3.14, 2.4, -.03, 0];
    let uniforms = { u_material : {}, 
                     u_light: { position: lightPosition, la: [0.1,0.1,0.1], ld: [1, 1, 1],
                                ls: [0.8, 0.8, 0.8]}};

    let normal_mapping =  new NormalMapping(gl, WebGL, planeFigure, srcTextures, setTextures, uniforms); 
    let no_normal_mapping =  new PhongTexture(gl, WebGL, planeFigure, srcTextures, setTextures, uniforms, {}); 

    let nm_plane = new GeometryPainter(gl, normal_mapping, M4.translate(0,-.1,0));
    let mat_plane = new GeometryPainter(gl, no_normal_mapping, M4.translate(0,-.1,0));
    var pos = new V3(0,0,.8);
    let camera = new TrackballCamera(pos);
    camera.setZoomConfig(2,.6);

    let bulb = new Light(gl, WebGL, lightPosition, 0.1);
    let textures;
    var translation;
    var modelMatrix;
    var modelViewMatrix;
    
    let light_controls = document.getElementById("ui-container-up");
    addLabel(light_controls, "Posición de la luz");
    var light_pos_x = new Slider(light_controls, "x", -10, 10, upadeteLPos(0), lightPosition[0], 0.01, ["#4994D0"]);
    var light_pos_y = new Slider(light_controls, "y", -10, 10, upadeteLPos(1), lightPosition[1], 0.01, ["#4994D0"]);
    var light_pos_z = new Slider(light_controls, "z", -3, 6, upadeteLPos(2), lightPosition[2], 0.01, ["#4994D0"]);
    
    let textureIds = { ladrillos: 0, baldosas: 2, azulejos: 4, grava: 6};
    /* Selects */
    let right_controls = document.getElementById("right-options");
    let selectTexture = new Select(right_controls, "textures", setTextureId, textureIds, "ladrillos");
    let normalMappinCheckbox = new Checkbox(right_controls, "normalmap", "Normal Map", draw, true);
    let darkMode = new DarkMode(draw, [light_controls]);

    let textureId = 0;
 
    gl.enable(gl.DEPTH_TEST);
    gl.enable(gl.CULL_FACE);
    
    gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
    gl.clearColor(0, 0, 0, 0);

    window.addEventListener('resize', draw);
    registerCameraEvents(camera, canvas, draw);
    
    function draw(){        
        resize(canvas);
        resizeAndCenterX(light_controls, {});
        darkMode.check(gl);
        
        var {width, height} = gl.canvas;
        
        gl.viewport(0, 0, width, height);
        gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
        
        let aspect = width/height;
        var projectionMatrix = M4.perspective(90, aspect, .1, 100);
        var viewMatrix = camera.getMatrix();
        var lightPos = M4.multiplyVector(viewMatrix, lightPosition);
    
        uniforms.u_light.position = lightPos;
        if(textures)
            uniforms.u_material = { diffuse_map : { id: textureId, texture: textures[textureId]},
                                    normal_map : { id: textureId+1, texture: textures[textureId+1]},
                                    shininess: 0.09*128};

        if(normalMappinCheckbox.checked()){
            nm_plane.draw(gl, viewMatrix, projectionMatrix, uniforms);
        }else{
            mat_plane.draw(gl, viewMatrix, projectionMatrix, uniforms);
        }      
         
        bulb.draw(gl, viewMatrix, projectionMatrix, lightPosition);
    }
    function setTextures(loadedTextures){
        textures = loadedTextures;
        draw();
    }
    function upadeteLPos(index){
        return function(value){
            lightPosition[index] = value;
            draw();
        }
    }
    function setTextureId(value){
        textureId = textureIds[value];
        draw();
    }
} 